<?php

defined('JPATH_BASE') or die();

/**
 * MSSQL database driver.
 * It's improved Joomla 1.5 MySQL driver for MSSQL.
 *
 * @package		ARTIOFusionCharts
 * @subpackage	Database
 * @since		1.0
 */

class JDatabaseMSSQL
{

    /**
     * Database server name.
     *
     * @var string
     * @access private
     */
    var $_host;

    /**
     * Database user name.
     *
     * @var string
     * @access private
     */
    var $_user;

    /**
     * Database password.
     *
     * @var string
     * @access private
     */
    var $_password;

    /**
     * Database name.
     *
     * @var string
     * @access private
     */
    var $_database;

    /**
     * Database tables prefix.
     *
     * @var string
     * @access private
     */
    var $_prefix;

    /**
     * Database connection.
     *
     * @var resource
     * @access private
     */
    var $_resource;

    /**
     * Latest SQL query.
     *
     * @var string
     * @access private
     */
    var $_sql;

    /**
     * Length of list of rows.
     *
     * @var int
     * @access private
     */
    var $_limit;
    /**
     * Offset of list of rows
     *
     * @var int
     * @access private
     */
    var $_offset;

    /**
     * Database object constructor
     *
     * @access	public
     * @param	array	List of options used to configure the connection
     * @since	1.5
     * @see		JDatabase
     */
    function __construct($options)
    {
        $this->_host = JArrayHelper::getValue($options, 'host', 'localhost', 'string');
        $this->_user = JArrayHelper::getValue($options, 'user', '', 'string');
        $this->_password = JArrayHelper::getValue($options, 'password', '', 'string');
        $this->_database = JArrayHelper::getValue($options, 'database', '', 'string');
        $this->_prefix = JArrayHelper::getValue($options, 'prefix', '', 'string');

        // perform a number of fatality checks, then return gracefully
        if (!function_exists('mssql_connect')) {
            $this->_errorNum = 1;
            $this->_errorMsg = 'The MSSQL adapter "mssql" is not available.';
            return;
        }

        // connect to the server
        $this->_resource = mssql_connect($this->_host, $this->_user, $this->_password, true);
        if (!$this->_resource) {
            $this->_errorNum = 2;
            $this->_errorMsg = 'Could not connect to MSSQL';
            return;
        }

        // select the database
        if (!mssql_select_db($this->_database, $this->_resource)) {
            $this->_errorNum = 2;
            $this->_errorMsg = 'Could not select database';
            return;
        }
    }

    /**
     * Sets the SQL query string for later execution.
     *
     * This function replaces a string identifier <var>$prefix</var> with the
     * string held is the <var>_table_prefix</var> class variable.
     *
     * @access public
     * @param string The SQL query
     * @param string The offset to start selection
     * @param string The number of results to return
     * @param string The common table prefix
     */
    function setQuery($sql, $offset = 0, $limit = 0, $prefix = '#__')
    {
        $this->_sql = $this->replacePrefix($sql, $prefix);
        $this->_limit = (int) $limit;
        $this->_offset = (int) $offset;
    }

    /**
     * This function replaces a string identifier <var>$prefix</var> with the
     * string held is the <var>_table_prefix</var> class variable.
     *
     * @access public
     * @param string The SQL query
     * @param string The common table prefix
     */
    function replacePrefix($sql, $prefix = '#__')
    {
        return str_replace($prefix, $this->_prefix, $sql);
    }

    /**
     * Execute the query
     *
     * @access	public
     * @return mixed A database resource if successful, FALSE if not.
     */
    function query()
    {
        if (!is_resource($this->_resource)) {
            return false;
        }

        // Take a local copy so that we don't modify the original query and cause issues later
        $sql = $this->_sql;
        if ($this->_limit > 0 || $this->_offset > 0) {
            $sql .= ' LIMIT ' . max($this->_offset, 0) . ', ' . max($this->_limit, 0);
        }

        $this->_errorNum = 0;
        $this->_errorMsg = '';
        $this->_cursor = mssql_query($sql, $this->_resource );

        if (!$this->_cursor) {
            $this->_errorNum = 500;
            $this->_errorMsg = mssql_get_last_message($this->_resource ) . " SQL=$sql";

            if ($this->_debug) {
                JError::raiseError(500, 'JDatabaseMSSQL::query: ' . $this->_errorNum . ' - ' . $this->_errorMsg );
            }
            return false;
        }
        return $this->_cursor;
    }

    /**
     * Load a list of database rows (numeric column indexing)
     *
     * @access public
     * @param string The field name of a primary key
     * @return array If <var>key</var> is empty as sequential list of returned records.
     * If <var>key</var> is not empty then the returned array is indexed by the value
     * the database key.  Returns <var>null</var> if the query fails.
     */
    function loadRowList($key = null)
    {
        if (!($cur = $this->query())) {
            return null;
        }
        $array = array();
        while ($row = mssql_fetch_row($cur)) {
            if ($key !== null) {
                $array[$row[$key]] = $row;
            } else {
                $array[] = $row;
            }
        }
        mssql_free_result($cur);
        return $array;
    }

}
?>