<?php

/**
 * Componet for creating Fusion Charts XML source in graphic mode.
 * Can create, modified and deleting assorted Fusion Charts. Charts
 * can display on frontend page by module or content plugin.  
 *
 * @version		$Id$
 * @package     ArtioFusioncharts
 * @copyright	Copyright (C) 2010 ARTIO s.r.o.. All rights reserved. 
 * @author 		ARTIO s.r.o., http://www.artio.net
 * @license     GNU/GPL http://www.gnu.org/copyleft/gpl.html
 * @link        http://www.artio.net Official website
 */

defined('_JEXEC') or die('Restricted access');

class ARTIOFusionChartsHelper
{

    /**
     * Load all available charts categories from charts directory /components/com_artiofusioncharts/charts/.
     * All subdirectory is one charts category containing charts types.
     * 
     * @return array
     */
    function loadCategories()
    {
        $types = JFolder::folders(ARTIOFUSIONCHARTS_BASE_ABS);
        $categories = array();
        foreach ($types as $type) {
            if (($category = ARTIOFusionChartsHelper::loadCategory($type)) !== null) {
                $categories[] = $category;
            }
        }
        return $categories;
    }

    /**
     * Load data and setting for concrete charts category from directory storage.
     * Category is loaded with all charts types.
     * 
     * @param string $type
     * @return stdClass|null
     */
    function loadCategory($type)
    {
        $file = ARTIOFusionChartsHelper::getChartMainCategoryParams($type);
        if (file_exists($file)) {
            require ($file);
            $category = new stdClass();
            $category->name = $categoryName;
            $category->base = $type;
            $charts = JFolder::folders(ARTIOFusionChartsHelper::getChartCategoryBase($type));
            $categoryCharts = array();
            foreach ($charts as $item) {
                if (($chart = ARTIOFusionChartsHelper::loadChart($item, $type)) !== null) {
                    $categoryCharts[] = $chart;
                }
            }
            $category->charts = $categoryCharts;
            return $category;
        }
        return null;
    }

    function getCategoryNamesArray($categories)
    {
        $names = array();
        foreach ($categories as $category) {
            $names[$category->base] = $category->name;
        }
        return $names;
    }

    function loadChart($item, $type, $chart = null)
    {
        $file = ARTIOFusionChartsHelper::getChartMainParams($type, $item);
        
        if (file_exists($file)) {
            
            include ($file);
            
            if (is_null($chart)) {
                $chart = new stdClass();
                $chart->xml = '';
            }
            
            $chart->name = $chartName;
            $chart->chartType = $chartType;
            $chart->chartLanguages = isset($chartLanguages) ? $chartLanguages : array();
            $chart->chartSetting = isset($chartSetting) ? $chartSetting : array();
            $chart->base = $item;
            
            $chart->icon = ARTIOFusionChartsHelper::getChartIcon($type, $item);
            $chart->preview = ARTIOFusionChartsHelper::getChartPrewiev($type, $item);
            $chart->chart = ARTIOFusionChartsHelper::getChartFile($type, $item);
            
            $chart->paramsSource = ARTIOFusionChartsHelper::getChartParams($type, $item);
            $chart->paramsCategoriesSource = ARTIOFusionChartsHelper::getChartCategoriesParams($type, $item);
            $chart->paramsCategorySource = ARTIOFusionChartsHelper::getChartCategoryParams($type, $item);
            $chart->paramsDatasetSource = ARTIOFusionChartsHelper::getChartDatasetParams($type, $item);
            $chart->paramsSetSource = ARTIOFusionChartsHelper::getChartSetParams($type, $item);
            $chart->paramsTrendLinesSource = ARTIOFusionChartsHelper::getChartTrendLinesParams($type, $item);
            $chart->paramsLinesetSource = ARTIOFusionChartsHelper::getChartLineSetParams($type, $item);
            
            $chart->params = ARTIOFusionChartsHelper::parseChartParams($chart->xml);
            $chart->jparams = ARTIOFusionChartsHelper::getJoomlaParamsFormat($chart->params);
            $chart->jparams = ARTIOFusionChartsHelper::quote($chart->jparams);
            
            ARTIOFusionChartsHelper::fillChartSetting($chart);
            
            $chart->form = new JParameter('', $chart->paramsSource);
            $chart->form->loadINI($chart->jparams);
            
            $chart->link = ARTIOFusionChartsHelperRoute::getCreateRoute($type, $item);
            
            $chart->sets = ARTIOFusionChartsHelper::parseChartSets($chart->xml);
            $chart->trendlines = ARTIOFusionChartsHelper::parseChartTrendlines($chart->xml);
            
            $chart->categoryParams = ARTIOFusionChartsHelper::parseChartCategoriesParams($chart->xml);
            
            $chart->categories = ARTIOFusionChartsHelper::parseChartCategories($chart->xml);
            
            $chart->datasets = ARTIOFusionChartsHelper::parseChartDatasets($chart->xml);
            
            $chart->linesets = ARTIOFusionChartsHelper::parseChartLinesets($chart->xml);
            
            if (isset($chart->useSqlQuery) && $chart->useSqlQuery == ARTIOFUSIONCHARTS_USE_SQL_QUERY_REAL_TIME) {
                ARTIOFusionChartsHelper::fillChartFromDB($chart);
            }
            
            return $chart;
        }
        
        return null;
    }

    function fillChartFromDB(&$chart)
    {
        $db = &ARTIOFusionChartsHelper::getDBO($chart->database);
        /* @var $db JDatabaseMySQL */
        $db->setQuery(str_replace('{JUSERID}', JFactory::getUser()->get('id'), $chart->sqlQuery));
        // data provided by set database query
        $data = &$db->loadRowList();
        $countData = count($data);
        if (! is_null($data)) {
            if ($chart->chartType == ARTIOFUSIONCHARTS_SINGLE_SERIES_TYPE) {
                // for all saved chart sets
                foreach ($chart->sets as $i => $set) {
                    // get all current set params to filling from database row
                    $params = &ARTIOFusionChartsHelper::getParamsAsArray($set);
                    $keys = array_keys($params);
                    // if available database row for this set
                    if (isset($data[$i])) {
                        // set row cols as set params values
                        foreach ($data[$i] as $j => $col) {
                            $col = JString::trim($col);
                            if ($col && isset($keys[$j])) {
                                $params[$keys[$j]] = ARTIOFusionChartsHelper::quote($col);
                            }
                        }
                        $chart->sets[$i] = ARTIOFusionChartsHelper::getArrayAsParams($params);
                    } else {
                        // if not available database row delete set
                        unset($chart->sets[$i]);
                    }
                }
                $countSets = count($chart->sets);
                // if database rows are more that saved sets
                if ($countData > $countSets) {
                    // get last saved set as mask
                    $set = end($chart->sets);
                    if ($set === false) {
                        // if haven't last set (saved chart is empty) create mask from XML source
                        $params = ARTIOFusionChartsHelper::getParamsArrayFromSource($chart->paramsSetSource);
                    } else {
                        // if have last set
                        $params = &ARTIOFusionChartsHelper::getParamsAsArray($set);
                    }
                    $keys = array_keys($params);
                    // for all database rows on top saved sets
                    for ($i = $countSets; $i < $countData; $i ++) {
                        foreach ($data[$i] as $j => $col) {
                            $col = JString::trim($col);
                            if ($col && isset($keys[$j])) {
                                $params[$keys[$j]] = ARTIOFusionChartsHelper::quote($col);
                            }
                        }
                        $newSet = ARTIOFusionChartsHelper::getArrayAsParams($params);
                        $actualEndSet = end($chart->sets);
                        // add color from poll
                        $newSet = &ARTIOFusionChartsHelper::setColorInParams($newSet, $actualEndSet === false ? '' : $actualEndSet);
                        $chart->sets[] = $newSet;
                    }
                }
            }
            if ($chart->chartType == ARTIOFUSIONCHARTS_MULTI_SERIES_TYPE) {
                if (isset($data[0])) {
                    // set first database row as categories
                    foreach ($chart->categories as $i => $category) {
                        // for all saved categories
                        $ci = $i + 1;
                        // first is ignored
                        if (isset($data[0][$ci])) {
                            $chart->categories[$i]->value = $data[0][$ci];
                        } else {
                            unset($chart->categories[$i]);
                        }
                    }
                    $countCategories = count($chart->categories);
                    $countCols = count($data[0]) - 1;
                    if ($countCols > $countCategories) {
                        // generate categories
                        for ($i = $countCategories; $i < $countCols; $i ++) {
                            $chart->categories[] = ARTIOFusionChartsHelper::createCategory($data[0][$i + 1], '');
                        }
                    }
                }
            }
            foreach ($chart->datasets as $i => $dataset) {
                $di = $i + 1;
                if (isset($data[$di])) {
                    if (isset($data[$di][0])) {
                        $chart->datasets[$i]->value = $data[$di][0];
                    }
                    foreach ($dataset->sets as $j => $set) {
                        $ji = $j + 1;
                        if (isset($data[$di][$ji])) {
                            $chart->datasets[$i]->sets[$j]->value = ARTIOFusionChartsHelper::quote($data[$di][$ji]);
                        } else {
                            unset($chart->datasets[$i]->sets[$j]);
                        }
                    }
                } else {
                    unset($chart->datasets[$i]);
                }
            }
            $countDatasets = count($chart->datasets);
            if ($countData > $countDatasets + 1) {
                $endDataset = end($chart->datasets);
                if ($endDataset === false) {
                    $endDataset = &ARTIOFusionChartsHelper::createDataset('', '');
                }
                for ($i = $countDatasets + 1; $i < $countData; $i ++) {
                    $dataset = &ARTIOFusionChartsHelper::createDataset($data[$i][0], $endDataset->params);
                    foreach ($data[$i] as $j => $value) {
                        if ($j) {
                            $dataset->sets[] = &ARTIOFusionChartsHelper::createSet($value, isset($endDataset->sets[$j - 1]) ? $endDataset->sets[$j - 1]->params : '');
                        }
                    }
                    $actualEndDataset = end($chart->datasets);
                    $dataset->params = &ARTIOFusionChartsHelper::setColorInParams($dataset->params, is_object($actualEndDataset) ? $actualEndDataset->params : '');
                    $chart->datasets[] = $dataset;
                }
            }
        } else {
            $mainframe = &JFactory::getApplication();
            /* @var $mainframe JApplication */
            $mainframe->enqueueMessage(JText::sprintf('SQL query used for fill chart "%s" is invalid. Chart filled by default saved data.', $chart->title), 'error');
        }
        $chart->xml = ARTIOFusionChartsHelper::getChartXML($chart);
    }

    function fillChartSetting(&$chart)
    {
        if (isset($chart->setting)) {
            $config = &JComponentHelper::getParams(ARTIOFUSIONCHARTS_OPTION);
            /* @var $config JParameter */
            $setting = new JParameter($chart->setting);
            $chart->height = (int) $setting->getValue('height', $config->get('new_chart_height', ARTIOFUSIONCHARTS_DEFAULT_WIDTH));
            $chart->width = (int) $setting->getValue('width', $config->get('new_chart_width', ARTIOFUSIONCHARTS_DEFAULT_WIDTH));
            $chart->useSqlQuery = $setting->getValue('useSqlQuery', ARTIOFUSIONCHARTS_USE_SQL_QUERY_NO);
            $chart->sqlQuery = $setting->getValue('sqlQuery', '');
            $chart->database = $setting->getValue('database', '');
        }
    }

    function loadChartSetting(&$data, $request = false)
    {
        $setting['height'] = (int) $data['height'];
        $setting['width'] = (int) $data['width'];
        $setting['useSqlQuery'] = (int) isset($data['useSqlQuery']) ? $data['useSqlQuery'] : null;
        if ($request)
            $setting['sqlQuery'] = (string) $_REQUEST['sqlQuery'] ? stripslashes($_REQUEST['sqlQuery']) : null;
        else
            $setting['sqlQuery'] = (string) isset($data['sqlQuery']) ? $data['sqlQuery'] : null;
        $setting['database'] = (int) isset($data['database']) ? $data['database'] : null;
        return $setting;
    }

    function parseChartParams($xml)
    {
        $match = array();
        if (preg_match('#\<' . CHART . ' (.*)\>(.*)\</' . CHART . '\>#iU', $xml, $match)) {
            return $match[1];
        }
        return '';
    }

    function parseChartSets($xml)
    {
        $matches = array();
        if (preg_match_all('#\<set (.*)/\>#iU', $xml, $matches, PREG_PATTERN_ORDER)) {
            return $matches[1];
        }
        return array();
    }

    function parseChartTrendlines($xml)
    {
        $matches = array();
        if (preg_match_all('#\<line (.*)/\>#iU', $xml, $matches, PREG_PATTERN_ORDER)) {
            return $matches[1];
        }
        return array();
    }

    function parseChartCategoriesParams($xml)
    {
        $match = array();
        if (preg_match('#\<categories (.*)\>(.*)\</categories\>#iU', $xml, $match)) {
            return $match[1];
        }
        return '';
    }

    function parseChartCategories($xml)
    {
        $matches = array();
        $categories = array();
        if (preg_match_all('#\<category name\=\'(.*)\' (.*)/\>#iU', $xml, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $categories[] = &ARTIOFusionChartsHelper::createCategory($match[1], $match[2]);
            }
        }
        return $categories;
    }

    function parseChartDatasets($xml)
    {
        $matches = array();
        $datasets = array();
        $xml = str_replace(array('<dataset><dataset','</dataset></dataset>'), array('<dataset', '</dataset>'), $xml);
        if (preg_match_all("#\<dataset seriesname\='(.*)'(.*)\>(.*)\</dataset\>#iU", $xml, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $dataset = &ARTIOFusionChartsHelper::createDataset($match[1], $match[2]);
                $matchesSets = array();
                if (preg_match_all('#\<set value\=\'(.*)\' (.*)/\>#iU', $match[3], $matchesSets, PREG_SET_ORDER)) {
                    foreach ($matchesSets as $matchSet) {
                        $dataset->sets[] = &ARTIOFusionChartsHelper::createSet(ARTIOFusionChartsHelper::quote($matchSet[1]), $matchSet[2]);
                    }
                }
                $datasets[] = $dataset;
            }
        }
        return $datasets;
    }
    
    function parseChartLinesets($xml)
    {
    	$matches = array();
    	$linesets = array();
    	if (preg_match_all("#\<lineset seriesname\='(.*)'(.*)\>(.*)\</lineset\>#iU", $xml, $matches, PREG_SET_ORDER)) {
    		foreach ($matches as $match) {
    			$lineset = &ARTIOFusionChartsHelper::createLineset($match[1], $match[2]);
    			$matchesSets = array();
    			if (preg_match_all('#\<set value\=\'(.*)\' (.*)/\>#iU', $match[3], $matchesSets, PREG_SET_ORDER)) {
    				foreach ($matchesSets as $matchSet) {
    					$lineset->sets[] = &ARTIOFusionChartsHelper::createSet(ARTIOFusionChartsHelper::quote($matchSet[1]), $matchSet[2]);
    				}
    			}
    			$linesets[] = $lineset;
    		}
    	}
    	return $linesets;
    }

    /**
     * Create and fill chart dataset object.
     * @param string $value
     * @param string $params
     * @return stdClass
     */
    function createDataset($value, $params, $group = null)
    {
        $dataset = &ARTIOFusionChartsHelper::createCommonObject($value, $params);
        $dataset->sets = array();
        $dataset->object = self::getParamsAsArray($params);
        if ($group) {
        	$dataset->object['datasetGroup'] = $group;
        	$dataset->params = self::getArrayAsParams($dataset->object) . ' ';
        }
        return $dataset;
    }
    
    /**
     * Create and fill chart lineset object.
     * @param string $value
     * @param string $params
     * @return stdClass
     */
    function createLineset($value, $params)
    {
    	$lineset = &ARTIOFusionChartsHelper::createCommonObject($value, $params);
    	$lineset->sets = array();
    	return $lineset;
    }

    /**
     * Create and fill chart set object.
     * @param string $value
     * @param string $params
     * @return stdClass
     */
    function createSet($value, $params)
    {
        $set = &ARTIOFusionChartsHelper::createCommonObject($value, $params);
        return $set;
    }

    /**
     * Create and fill chart category object.
     * @param string $value
     * @param string $params
     * @return stdClass
     */
    function createCategory($value, $params)
    {
        $category = &ARTIOFusionChartsHelper::createCommonObject($value, $params);
        return $category;
    }

    /**
     * Create and fill chart common object.
     * @param string $value
     * @param string $params
     * @return stdClass
     */
    function createCommonObject($value, $params)
    {
        $object = new stdClass();
        $object->value = ARTIOFusionChartsHelper::unquote($value);
        $object->params = $params;
        return $object;
    }

    function getChartXMLFromRequest(&$data)
    {
        $chart = new stdClass();
        
        $chart->params = isset($data['params']) ? $data['params'] : array();
        
        $chart->categoryParams = isset($data['categoryParams']) ? $data['categoryParams'] : array();
        
        $chart->categories = array();
        
        $categoriesValues = isset($data['chart-category']) ? $data['chart-category'] : array();
        $categoriesParams = isset($data['chart-category-params']) ? $data['chart-category-params'] : array();
        
        foreach ($categoriesValues as $i => $categoryValue) {
            $chart->categories[] = &ARTIOFusionChartsHelper::createCategory($categoryValue, $categoriesParams[$i]);
        }
        
        $datasetsValues = isset($data['dataset']) ? $data['dataset'] : array();
        $datasetsParams = isset($data['dataset-params']) ? $data['dataset-params'] : array();
        $datasetsGroups = isset($data['dataset-group']) ? $data['dataset-group'] : array();
        
        $chart->datasets = array();
        
        foreach ($datasetsValues as $i => $datasetValue) {
            $dataset = &ARTIOFusionChartsHelper::createDataset($datasetValue, $datasetsParams[$i], $datasetsGroups[$i]);
            
            $setValues = isset($data['dataset-' . $i . '-values']) ? $data['dataset-' . $i . '-values'] : array();
            $setParams = isset($data['dataset-' . $i . '-value-params']) ? $data['dataset-' . $i . '-value-params'] : array();
            
            foreach ($setValues as $j => $setValue) {
                $dataset->sets[] = &ARTIOFusionChartsHelper::createSet($setValue, $setParams[$j]);
            }
            
            $chart->datasets[] = $dataset;
        }
        
        $sets = isset($data['sets']) ? $data['sets'] : array();
        $chart->sets = ARTIOFusionChartsHelper::getParamsFromRequest($sets);
        
        $trendlines = isset($data['trendlines']) ? $data['trendlines'] : array();
        $chart->trendlines = ARTIOFusionChartsHelper::getParamsFromRequest($trendlines);
        
        $linesetsValues = isset($data['lineset']) ? $data['lineset'] : array();
        $linesetsParams = isset($data['lineset-params']) ? $data['lineset-params'] : array();
        
        $chart->linesets = array();
        
        foreach ($linesetsValues as $i => $linesetValue) {
        	$lineset = &ARTIOFusionChartsHelper::createLineset($linesetValue, $linesetsParams[$i]);
        
        	$setValues = isset($data['lineset-' . $i . '-values']) ? $data['lineset-' . $i . '-values'] : array();
        
        	foreach ($setValues as $j => $setValue) {
        		$lineset->sets[] = &ARTIOFusionChartsHelper::createSet($setValue, '');
        	}
        
        	$chart->linesets[] = $lineset;
        }
        
        $xml = ARTIOFusionChartsHelper::getChartXML($chart);
        
        return $xml;
    }

    function getChartXML(&$chart)
    {
        $xml = '<' . CHART;
        
        if (is_array($chart->params)) {
            foreach ($chart->params as $param => $value) {
                $xml .= ' ' . $param . "='" . ARTIOFusionChartsHelper::quote($value) . "'";
            }
        } else {
            $xml .= ' ' . $chart->params;
        }
        
        $xml .= '>';
        
        if (count($chart->categories)) {
            $xml .= '<categories ' . $chart->categoryParams . '>';
            foreach ($chart->categories as $category) {
                $xml .= "<category name='" . ARTIOFusionChartsHelper::quote($category->value) . "' " . $category->params . '/>';
            }
            $xml .= '</categories>';
        }
        
        $groups = array();
        
        foreach ($chart->datasets as $dataset) {
            $dxml = "<dataset seriesname='" . ARTIOFusionChartsHelper::quote($dataset->value) . "' " . $dataset->params . '>';
            foreach ($dataset->sets as $set) {
                $dxml .= "<set value='" . ARTIOFusionChartsHelper::quote($set->value) . "' " . $set->params . ' />';
            }
            $dxml .= '</dataset>';
            if ($dataset->object['datasetGroup'])
            	$groups[$dataset->object['datasetGroup']][] = $dxml;
            else
            	$xml .= $dxml;
        }
        
        foreach ($groups as $id => $datasets)
        	$xml .= '<dataset>' . implode('', $datasets) . '</dataset>';
        
        foreach ($chart->sets as $set) {
            $xml .= '<set ' . $set . '/>';
        }
        
        if (count($chart->trendlines)) {
            $xml .= '<trendLines>';
            foreach ($chart->trendlines as $trendline) {
                $xml .= '<line ' . $trendline . '/>';
            }
            $xml .= '</trendLines>';
        }
        
        foreach ($chart->linesets as $lineset) {
        	$xml .= "<lineset seriesname='" . ARTIOFusionChartsHelper::quote($lineset->value) . "' " . $lineset->params . '>';
        	foreach ($lineset->sets as $set)
        		$xml .= "<set value='" . ARTIOFusionChartsHelper::quote($set->value) . "' " . $set->params . ' />';
        	$xml .= '</lineset>';
        }
        
        $xml .= '</' . CHART . '>';
        
        return $xml;
    }

    /**
     * Convert standard chart params format like: param1='value1' param2='value2 into Joomla storage format like:
     * param1=value1
     * param2=value2
     * 
     * @param string $params
     * @return string can be empty if did'nt find any param
     */
    function getJoomlaParamsFormat($params)
    {
        $output = '';
        $mustEncoded = &ARTIOFusionChartsHelper::mustEncoded();
        $params = &ARTIOFusionChartsHelper::getParamsAsArray($params);
        foreach ($params as $param => $value) {
            if (in_array($param, $mustEncoded)) {
                $value = ARTIOFusionChartsHelper::encode($value);
            }
            $output .= $param . '=' . $value . "\n";
        }
        return $output;
    }

    /**
     * Get standard chart params format like: param1='value1' param2='value2'
     * as array like: $output[param1] = value1, $output[param2] = value2
     * 
     * @param string $params
     * @return array can be empty if did'nt find any param
     */
    function getParamsAsArray($params)
    {
        $output = array();
        $matches = array();
        if (preg_match_all("#(.*)\='(.*)'#iU", $params, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $output[JString::trim($match[1])] = JString::trim($match[2]);
            }
        }
        return $output;
    }

    /**
     * Get standard chart params format like: param1='value1' param2='value2'
     * from array like: $output[param1] = value1, $output[param2] = value2
     * 
     * @param array $array
     * @return string can be empty if did'nt find any param
     */
    function getArrayAsParams($array)
    {
        $output = '';
        foreach ($array as $param => $value) {
            $output .= $param . '=\'' . $value . '\' ';
        }
        return $output;
    }

    function getChartParamsFormat($params)
    {
        $params = str_replace("\n", ' ', $params);
        return $params;
    }

    
    function getTypesNamesArray($categories)
    {
        $names = array();
        foreach ($categories as $category) {
            $namesTypes = ARTIOFusionChartsHelper::getChartsNamesArray($category);
            $names = array_merge($names, $namesTypes);
        }
        return $names;
    }

    function getChartsNamesArray($category)
    {
        $names = array();
        foreach ($category->charts as $type) {
            $names[$type->base] = $type->name;
        }
        return $names;
    }

    function mustEncoded()
    {
        return array('numberSuffix' , 'numberPrefix');
    }

    function encode($string)
    {
        $chTable = array(' ' => '%20' , '!' => '%21' , '"' => '%22' , '#' => '%23' , '$' => '%24' , '%' => '%25' , '&' => '%26' , '\'' => '%28' , '(' => '%28' , ')' => '%29' , '*' => '%2A' , '+' => '%2B' , ',' => '%2C' , '-' => '%2D' , '.' => '%2E' , '/' => '%2F' , '0' => '%30' , '1' => '%31' , '2' => '%32' , '3' => '%33' , '4' => '%34' , '5' => '%35' , '6' => '%36' , '7' => '%37' , '8' => '%38' , '9' => '%39' , ':' => '%3A' , ';' => '%3B' , '<' => '%3C' , '=' => '%3D' , '>' => '%3E' , '?' => '%3F' , '@' => '%40' , 'A' => '%41' , 'B' => '%42' , 'C' => '%43' , 'D' => '%44' , 'E' => '%45' , 'F' => '%46' , 'G' => '%47' , 'H' => '%48' , 'I' => '%49' , 'J' => '%4A' , 'K' => '%4B' , 'L' => '%4C' , 'M' => '%4D' , 'N' => '%4E' , 'O' => '%4F' , 'P' => '%50' , 'Q' => '%51' , 'R' => '%52' , 'S' => '%53' , 'T' => '%54' , 'U' => '%55' , 'V' => '%56' , 'W' => '%57' , 'X' => '%58' , 'Y' => '%59' , 'Z' => '%5A' , '[' => '%5B' , '\\' => '%5C' , ']' => '%5D' , '^' => '%5E' , '_' => '%5F' , '`' => '%60' , 'a' => '%61' , 'b' => '%62' , 'c' => '%63' , 'd' => '%64' , 'e' => '%65' , 'f' => '%66' , 'g' => '%67' , 'h' => '%68' , 'i' => '%69' , 'j' => '%6A' , 'k' => '%6B' , 'l' => '%6C' , 'm' => '%6D' , 'n' => '%6E' , 'o' => '%6F' , 'p' => '%70' , 'q' => '%71' , 'r' => '%72' , 's' => '%73' , 't' => '%74' , 'u' => '%75' , 'v' => '%76' , 'w' => '%77' , 'x' => '%78' , 'y' => '%79' , 'z' => '%7A' , '{' => '%7B' , '|' => '%7C' , '}' => '%7D' , '~' => '%7E' , '' => '%7F' , '€' => '%80' , '' => '%81' , '‚' => '%82' , 'ƒ' => '%83' , '„' => '%84' , '…' => '%85' , '†' => '%86' , '‡' => '%87' , 'ˆ' => '%88' , '‰' => '%89' , 'Š' => '%8A' , '‹' => '%8B' , 'Œ' => '%8C' , '' => '%8D' , 'Ž' => '%8E' , '' => '%8F' , '' => '%90' , '‘' => '%91' , '’' => '%92' , '“' => '%93' , '”' => '%94' , '•' => '%95' , '–' => '%96' , '—' => '%97' , '˜' => '%98' , '™' => '%99' , 'š' => '%9A' , '›' => '%9B' , 'œ' => '%9C' , '' => '%9D' , 'ž' => '%9E' , 'Ÿ' => '%9F' , '' => '%A0' , '¡' => '%A1' , '¢' => '%A2' , '£' => '%A3' , '' => '%A4' , '¥' => '%A5' , '|' => '%A6' , '§' => '%A7' , '¨' => '%A8' , '©' => '%A9' , 'ª' => '%AA' , '«' => '%AB' , '¬' => '%AC' , '¯' => '%AD' , '®' => '%AE' , '¯' => '%AF' , '°' => '%B0' , '±' => '%B1' , '²' => '%B2' , '³' => '%B3' , '´' => '%B4' , 'µ' => '%B5' , '¶' => '%B6' , '·' => '%B7' , '¸' => '%B8' , '¹' => '%B9' , 'º' => '%BA' , '»' => '%BB' , '¼' => '%BC' , '½' => '%BD' , '¾' => '%BE' , '¿' => '%BF' , 'À' => '%C0' , 'Á' => '%C1' , 'Â' => '%C2' , 'Ã' => '%C3' , 'Ä' => '%C4' , 'Å' => '%C5' , 'Æ' => '%C6' , 'Ç' => '%C7' , 'È' => '%C8' , 'É' => '%C9' , 'Ê' => '%CA' , 'Ë' => '%CB' , 'Ì' => '%CC' , 'Í' => '%CD' , 'Î' => '%CE' , 'Ï' => '%CF' , 'Ð' => '%D0' , 'Ñ' => '%D1' , 'Ò' => '%D2' , 'Ó' => '%D3' , 'Ô' => '%D4' , 'Õ' => '%D5' , 'Ö' => '%D6' , '' => '%D7' , 'Ø' => '%D8' , 'Ù' => '%D9' , 'Ú' => '%DA' , 'Û' => '%DB' , 'Ü' => '%DC' , 'Ý' => '%DD' , 'Þ' => '%DE' , 'ß' => '%DF' , 'à' => '%E0' , 'á' => '%E1' , 'â' => '%E2' , 'ã' => '%E3' , 'ä' => '%E4' , 'å' => '%E5' , 'æ' => '%E6' , 'ç' => '%E7' , 'è' => '%E8' , 'é' => '%E9' , 'ê' => '%EA' , 'ë' => '%EB' , 'ì' => '%EC' , 'í' => '%ED' , 'î' => '%EE' , 'ï' => '%EF' , 'ð' => '%F0' , 'ñ' => '%F1' , 'ò' => '%F2' , 'ó' => '%F3' , 'ô' => '%F4' , 'õ' => '%F5' , 'ö' => '%F6' , '÷' => '%F7' , 'ø' => '%F8' , 'ù' => '%F9' , 'ú' => '%FA' , 'û' => '%FB' , 'ü' => '%FC' , 'ý' => '%FD' , 'þ' => '%FE' , 'ÿ' => '%FF');
        $length = JString::strlen($string);
        $encoded = '';
        for ($i = 0; $i < $length; $i ++) {
            $char = JString::substr($string, $i, 1);
            $encoded .= isset($chTable[$char]) ? $chTable[$char] : $char;
        }
        return $encoded;
    }

    function getChart($id)
    {
        $model = &JModel::getInstance('artiofusionchart', 'artiofusionchartsmodel');
        /* @var $model ARTIOFusionChartsModelARTIOFusionChart */
        
        $chart = &$model->getData($id);
        /* @var $chart TableARTIOFusionChart */
        ARTIOFusionChartsHelper::loadChart($chart->type, $chart->category, $chart);
        //var_dump($chart);
        if (! is_null($chart->title)) {
            ARTIOFusionChartsHelper::fillChartSetting($chart);
            $chart->file = ARTIOFusionChartsHelper::getChartFile($chart->category, $chart->type);
            return $chart;
        }
        return false;
    }

    function displayChart(&$chart, $mask)
    {
        if ($chart) {
            static $count;
            if (is_null($count)) {
                $count = 1;
            }
            $count ++;
            $primaryID = $mask . '-chart-' . $count;
            $secondaryID = $mask . '-chartdiv-' . $count;
            $document = &JFactory::getDocument();
            /* @var $document JDocument */
            $js = '	   var chartObj = null;' . PHP_EOL;
            $js .= '	window.addEvent(\'load\', function() {' . PHP_EOL;
            $js .= '		chartObj = new FusionCharts("' . $chart->file . '", "ChartId", "' . $chart->width . '", "' . $chart->height . '", "0", "0");' . PHP_EOL;
            $js .= '		chartObj.setDataXML("' . $chart->xml . '");' . PHP_EOL;
            $js .= '            chartObj.addParam("wmode", "opaque");' . PHP_EOL;
            $js .= '            chartObj.setTransparent(true);' . PHP_EOL;
            $js .= '		chartObj.render("' . $secondaryID . '");' . PHP_EOL;
            $js .= '	});' . PHP_EOL;
            $document->addScriptDeclaration($js);
            return '<div id="' . $primaryID . '"><div id="' . $secondaryID . '"></div></div>';
        }
        return '<div>' . JText::_('Chart not found') . '</div>';
    }

    function exportParamsInJs($source, $var)
    {
        $xml = &JFactory::getXMLParser('Simple');
        /* @var $xml JSimpleXML */
        if ($xml->loadFile($source)) {
            $root = $xml->document;
            /* @var $root JSimpleXMLElement */
            $width = (int) $root->attributes('width');
            if (! $width)
                $width = 400;
            $height = (int) $root->attributes('height');
            if (! $height)
                $height = 400;
        }
        
        $datasetParams = new JParameter('', $source);
        
        $js = '<fieldset class="adminform"><table width="100%" class="paramlist admintable" cellspacing="1">';
        foreach ($datasetParams->getGroups() as $name => $count)
            foreach ($datasetParams->getParams('params', $name) as $param)
                $js .= '<tr><td width="40%" class="paramlist_key"><span class="editlinktip">' . $param[0] . '</span></td><td class="paramlist_value">' . $param[1] . '</td></tr>';
        $js .= '</table></fieldset>';
        
        $js = ARTIOFusionChartsHelper::quote($js, true);
        $js = '//<![CDATA[' . PHP_EOL . 'var ' . $var . " = '" . str_replace(array("\r\n" , "\n"), '', $js) . "'; " . PHP_EOL . '//]]>';
        
        $document = &JFactory::getDocument();
        /* @var $document JDocument */
        
        $document->addScriptDeclaration($js);
        $document->addScriptDeclaration('var ' . $var . 'Width = ' . $width . ';');
        $document->addScriptDeclaration('var ' . $var . 'Height = ' . $height . ';');
    }

    /**
     * Get simple array of params from standard XML source.
     * 
     * @param string $source
     * @return array where key is name of param, value is empty string
     */
    function getParamsArrayFromSource($source)
    {
        $output = array();
        $params = ARTIOFusionChartsHelper::getSourceParams($source);
        foreach ($params as $param) {
            $param = JString::trim($param[5]);
            $output[$param] = '';
        }
        return $output;
    }

    /**
     * Get params array from standard params XML source.
     * 
     * @param string $source XML source
     * @return array
     */
    function getSourceParams($source)
    {
        $params = new JParameter('', $source);
        
        // Work only with first group of params
        $group = $params->getGroups();
        $group = array_keys($group);
        $group = reset($group);
        
        $params = $params->getParams('params', $group);
        
        return $params;
    }

    function getParamsInTableForm($source, $mask)
    {
        $params = ARTIOFusionChartsHelper::getSourceParams($source);
        
        $output = new stdClass();
        $output->tableForms = array();
        $output->html = '<tr><td><input type="checkbox" name="checkbox[]" value=""/></td>';
        
        foreach ($params as $param) {
            $tableForm = new stdClass();
            
            $tableForm->title = JText::_($param[3]);
            $tableForm->desc = JText::_($param[2]);
            $tableForm->field = $param[5];
            $tableForm->html = $param[1];
            
            if (JString::strpos($tableForm->html, '<select') === 0) {
                //Is selectbox, search options
                $matches = array();
                if (preg_match_all('#\<option value\="([^"]*)"([^\>]*)\>#iU', $tableForm->html, $matches, PREG_SET_ORDER)) {
                    foreach ($matches as $match) {
                        // Add mask to add param checked="checked" like [%showOnTop1checked%]
                        $newOption = '<option value="' . $match[1] . '" ' . ARTIOFusionChartsHelper::getOptionCheckedMask($tableForm->field, $match[1]) . '>';
                        $tableForm->html = str_replace($match[0], $newOption, $tableForm->html);
                    }
                }
            } else {
                // Is text, add mask to replace value in param value=""
                $tableForm->html = str_replace('value=""', ARTIOFusionChartsHelper::getValueMask($tableForm->field), $tableForm->html);
            }
            // Improve field name, for example from name="params[alpha]" to  name="trendlines[alpha][]"
            $tableForm->html = str_replace('params[' . $tableForm->field . ']', $mask . '[' . $tableForm->field . '][]', $tableForm->html);
            
            // Remove ID is unused
            $tableForm->html = str_replace('id="params' . $tableForm->field . '"', '', $tableForm->html);
            
            $output->tableForms[] = $tableForm;
            // Prepare row HTML
            $output->html .= '<td>' . $tableForm->html . '</td>';
        }
        $output->html .= '</tr>';
        $output->html = str_replace('class="text_area"', 'style="width: 100%"', $output->html);
        
        return $output;
    }

    function getOptionCheckedMask($field, $value)
    {
        $mask = '[%' . $field . $value . 'checked%]';
        return $mask;
    }

    function getValueMask($field)
    {
        $mask = 'value="[%' . $field . '%]"';
        return $mask;
    }

    function getFillTableRowFromParams($row, $params)
    {
        $matches = array();
        // Parse params format like property=value
        if (preg_match_all("#(.*)\='(.*)'#iU", $params, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                // If is text input
                $match[1] = JString::trim($match[1]);
                $match[2] = JString::trim($match[2]);
                $row = str_replace(ARTIOFusionChartsHelper::getValueMask($match[1]), 'value="' . $match[2] . '"', $row);
                // If is select box
                $row = str_replace(ARTIOFusionChartsHelper::getOptionCheckedMask($match[1], $match[2]), 'selected="selected"', $row);
            }
        }
        $row = ARTIOFusionChartsHelper::cleanMasks($row);
        return $row;
    }

    function getMaskRow($row, $id)
    {
        $row = str_replace('<tr>', '<tr style="display: none;" id="' . $id . '">', $row);
        $row = ARTIOFusionChartsHelper::cleanMasks($row);
        return $row;
    }

    function cleanMasks($row)
    {
        $row = preg_replace('#\[%([^%]*)%\]#iU', '', $row);
        return $row;
    }

    function getParamsFromRequest(&$data)
    {
        $params = array();
        foreach ($data as $param => $items) {
            if (count($items)) {
                // First item is from row mask
                unset($items[0]);
            }
            foreach ($items as $i => $item) {
                if (! isset($params[$i - 1])) {
                    $params[$i - 1] = '';
                }
                $params[$i - 1] .= $param . "='" . ARTIOFusionChartsHelper::quote($item) . "' ";
            }
        }
        return $params;
    }

    function getChartEmptyCategories(&$chart, $count)
    {
        $chart->categories = array();
        for ($i = 0; $i < $count; $i ++) {
            $chart->categories[] = ARTIOFusionChartsHelper::getEmptyObject(JText::_('Category') . ' ' . ($i + 1), true);
        }
    }

    function getChartEmptySet(&$chart, $columnCount)
    {
        $chart->sets = array();
        for ($i = 0; $i < $columnCount; $i ++) {
            $item = "label='" . htmlentities($chart->chartSetting['sampleMask'] . ' ' . ($i + 1), ENT_QUOTES) . "' ";
            $item .= "value='" . rand(50, 150) . "' ";
            $item .= "color='" . ARTIOFusionChartsHelper::getColor() . "' ";
            $item .= "alpha='100' ";
            $item .= "htext='' ";
            $item .= "link='' ";
            $item .= "show='1' ";
            $item .= "isSliced='" . rand(0, 1) . "'";
            $chart->sets[] = $item;
        }
    }

    function getChartEmptyDataset(&$chart, $categoryCount, $datasetCount, $datasetGroupCount)
    {
        $chart->datasets = array();
        $group = 1;
        for ($i = 0; $i < $datasetCount; $i ++) {
            $dataset = ARTIOFusionChartsHelper::getEmptyObject(JText::_('Dataset') . ' ' . ($i + 1));
            $dataset->sets = array();
            if (!empty($chart->chartSetting['datasetGroup']))
            	$dataset->object['datasetGroup'] = $group;
            for ($j = 0; $j < $categoryCount; $j ++) {
                $dataset->sets[] = ARTIOFusionChartsHelper::getEmptyObject(rand(50, 150), true);
            }
            $chart->datasets[] = $dataset;
            if ($i % 2 && $i)
            	$group ++;
        }
    }
    
    function getChartEmptyLineset(&$chart) 
    {
    	$chart->linesets = array();
    	for ($i = 0; $i < 2; $i ++) {
    		$lineset = ARTIOFusionChartsHelper::getEmptyObject(JText::_('Lineset') . ' ' . ($i + 1));
    		$lineset->sets = array();
    		for ($j = 0; $j < 4; $j ++) {
    			$lineset->sets[] = ARTIOFusionChartsHelper::getEmptyObject(rand(50, 150), true);
    		}
    		$chart->linesets[] = $lineset;
    	}
    }

    function getEmptyObject($value, $noColor = false)
    {
        $object = new stdClass();
        $object->value = $value;
        $object->params = '';
        if (! $noColor) {
            $object->params = "color='" . ARTIOFusionChartsHelper::getColor() . "'";
        }
        return $object;
    }

    function getColor()
    {
        static $poll;
        if (is_null($poll)) {
            $poll = &ARTIOFusionChartsHelper::getColorsPoll();
        }
        static $index;
        if (is_null($index) || ($index == (count($poll) - 1))) {
            $index = 0;
        } else {
            $index ++;
        }
        $color = isset($poll[$index]) ? $poll[$index] : '';
        return $color;
    }

    /**
     * Set color in chart params string by values from before item from colors pool.
     * 
     * @param string $nextParams
     * @param string $beforeParams
     * @return string
     */
    function setColorInParams($nextParams, $beforeParams)
    {
        $nextParams = &ARTIOFusionChartsHelper::getParamsAsArray($nextParams);
        $beforeParams = &ARTIOFusionChartsHelper::getParamsAsArray($beforeParams);
        if (isset($nextParams['color'])) {
            $colors = &ARTIOFusionChartsHelper::getColorsPoll();
            $beforeColor = isset($beforeParams['color']) ? $beforeParams['color'] : false;
            $key = (int) array_search($beforeColor, $colors);
            $nextParams['color'] = $colors[++ $key % count($colors)];
        }
        $nextParams = &ARTIOFusionChartsHelper::getArrayAsParams($nextParams);
        return $nextParams;
    }

    function getColorsPoll()
    {
        static $colors;
        if (is_null($colors)) {
            $colors = array('AFD8F8' , 'F6BD0F' , '8BBA00' , 'FF8E46' , '008E8E' , 'D64646' , '8E468E' , '588526' , 'B3AA00' , '008ED6' , '9D080D' , 'A186BE' , 'CC6600' , 'FDC689' , 'ABA000' , 'F26D7D' , 'FFF200' , '0054A6' , 'F7941C' , 'CC3300' , '006600' , '663300' , '6DCFF6');
        }
        return $colors;
    }

    function importColorsToJS()
    {
        $colors = &ARTIOFusionChartsHelper::getColorsPoll();
        foreach ($colors as $i => $color) {
            $colors[$i] = '"' . $color . '"';
        }
        $js = 'var afcolors = new Array(' . implode(',', $colors) . ');';
        $document = &JFactory::getDocument();
        /* @var $document JDocument */
        $document->addScriptDeclaration($js);
    }

    function importJoomlaJS()
    {
        if (! file_exists(JPATH_ROOT . DS . 'includes' . DS . 'js' . DS . 'joomla.javascript.js'))
            JHTML::script('joomla.javascript.js?v=119', ARTIOFUSIONCHARTS_ASSETS_JS);
    }

    function quote($string, $onlyDoubleQuote = false)
    {
        if (! $onlyDoubleQuote)
            $string = str_replace('"', '&#034;', $string);
        return str_replace("'", '&#039;', $string);
    }

    function unquote($string)
    {
        return str_replace(array('&#034;' , '&#039;'), array('"' , "'"), $string);
    }

    function importXMl($xml)
    {
        $xml = str_replace(array("\r\n" , "\n" , "\t"), '', $xml);
        $xml = str_replace(array('<graph' , '</graph>'), array('<chart' , '</chart>'), $xml);
        $xml = JString::substr($xml, (int) JString::strpos($xml, '<chart'));
        return $xml;
    }

    function trySqlQuery($sqlQuery, $database)
    {
        $db = &ARTIOFusionChartsHelper::getDBO($database);
        /* @var $db JDatabase */
        $db->setQuery(str_replace('{JUSERID}', JFactory::getUser()->get('id'), $sqlQuery));
        $rows = &$db->loadRowList();
        if (is_null($rows)) {
            return 'SQL QUERY FAILED|' . $db->getErrorMsg(true);
        } elseif (is_array($rows) && count($rows) == 0) {
            return 'SQL QUERY EMPTY';
        } elseif (is_array($rows)) {
            foreach ($rows as $i => $row) {
                $rows[$i] = ARTIOFusionChartsHelper::getRowFormat($row);
            }
            return 'SQL QUERY OK|' . ARTIOFusionChartsHelper::getTableFormat($rows);
        } else {
            return 'SQL QUERY ERROR|' . JText::_('Internal server error');
        }
    }

    /**
     * Create database connector.
     * 
     * @param int $database 
     * @return JDatabase
     */
    function getDBO($database)
    {
        $mainframe = &JFactory::getApplication();
        /* @var $mainframe JApplication */
        $model = &JModel::getInstance('database', 'ARTIOFusionChartsModel');
        /* @var $model ARTIOFusionChartsModelDatabase */
        $database = &$model->getData($database);
        /* @var $database TableDatabase */
        
        if (! $database->id) {
            $dbo = &JFactory::getDBO();
            /* @var $dbo JDatabaseMySQL */
            return $dbo;
        }
        
        $options['driver'] = $database->type;
        $options['host'] = $database->host;
        $options['user'] = $database->user;
        $options['password'] = $database->password;
        $options['database'] = $database->database;
        $options['prefix'] = $database->prefix;
        
        if ($database->type == 'mssql') {
        	require_once JPATH_ROOT . '/administrator/components/com_artiofusioncharts/helpers/mssql.php';
        	$dbo = new JDatabaseMSSQL($options);
        	return $dbo;
        }
        
        $dbo = &JDatabase::getInstance($options);
        /* @var $dbo JDatabaseMySQL */
        
        if (JError::isError($dbo))
            $mainframe->enqueueMessage(JText::sprintf('Unable connect to database %s'), $database->title);
        
        return $dbo;
    }

    /**
     * Import data from file source into format to fill chart by javascript.
     * 
     * @param string $file filepath to source
     * @param int $sourceType set source type (XML,CSV)
     * @return string|booelan if return false - import failed
     */
    function importSource($file, $sourceType)
    {
        $mainframe = &JFactory::getApplication();
        /* @var $mainframe JApplication */
        if (function_exists('mime_content_type'))
            $mimeType = mime_content_type($file);
        elseif (function_exists('finfo_open') && function_exists('finfo_file') && function_exists('finfo_close')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file);
            finfo_close($finfo);
        }
        $table = false;
        if (isset($mimeType) && $mimeType !== 'text/plain') {
            $mainframe->enqueueMessage(JText::_('File is propably no CSV'), 'notice');
        } else {
            include (ARTIOFUSIONCHARTS_INCLUDES . 'parsecsv' . DS . 'parsecsv.lib.php');
            $csv = new parseCSV();
            $csv->auto($file);
            $rows = array();
            $cols = reset($csv->data);
            if (is_array($cols)) {
                $rows[] = ARTIOFusionChartsHelper::getRowFormat(array_keys($cols));
            }
            foreach ($csv->data as $row) {
                if (is_array($row) && count($row)) {
                    $rows[] = ARTIOFusionChartsHelper::getRowFormat($row);
                }
            }
            if (empty($rows)) {
                $mainframe->enqueueMessage(JText::_('Source file is empty'), 'notice');
            } else {
                $mainframe->enqueueMessage(JText::sprintf('Import %s rows', count($rows)), 'message');
                $table = ARTIOFusionChartsHelper::getTableFormat($rows);
            }
        }
        unlink($file);
        return $table;
    }

    /**
     * Get data row format. Row array as $row[]=value1, $row[]=value2 returned as string value1,value2.
     * Horizontal rule and comma replaced by HTML entities.
     * 
     * @param array $row
     * @return string
     */
    function getRowFormat($row)
    {
        foreach ($row as $i => $col) {
            $row[$i] = str_replace(array('|' , '-'), array('&brvbar;' , '&ndash;'), $col);
        }
        $row = implode('-', $row);
        return $row;
    }

    /**
     * Get data rows format. String rows separated by horizontal rule.
     * 
     * @param array $rows
     * @return string
     */
    function getTableFormat($rows)
    {
        $table = implode('|', $rows);
        return $table;
    }

    function getActions()
    {
        $user = &JFactory::getUser();
        /* @var $user JUser */
        $result = new JObject();
        foreach (array('core.admin' , 'core.manage' , 'core.create' , 'core.edit' , 'core.delete') as $action)
            $result->set($action, $user->authorise($action, ARTIOFUSIONCHARTS_OPTION));
        return $result;
    }

    function noauth()
    {
        $mainframe = &JFactory::getApplication();
        /* @var $mainframe JApplication */
        $mainframe->redirect(ARTIOFusionChartsHelperRoute::getHomeRoute(), JText::_('JERROR_ALERTNOAUTHOR'), 'error');
    }

    function canEdit($id)
    {
        if (($id && ! ARTIOFUSIONCHARTS_EDIT) || ! ARTIOFUSIONCHARTS_CREATE)
            ARTIOFusionChartsHelper::noauth();
    }

    /**
     * Replace link with top anchor <a href="#"> by link with concrete anchor.
     * For example: given ID is topContent. <a href="#"> will be replaced by <a href="#topContent"> everywhere.  
     *
     * @param string $html HTML containing anchors
     * @param string $id new ID to replace top anchor
     * @return string
     */
    function replaceTopAnchor($html, $id)
    {
        return str_replace('<a href="#">', '<a href="#' . $id . '">', $html);
    }

    /**
     * Decode URL param.
     * 
     * @param string $param
     * @return string
     */
    function decodeParam($param)
    {
        $decodes['____PLUS____'] = '+';
        $decodes['____LESSTHAN____'] = '<';
        $decodes['____GREATERTHAN____'] = '>';
        $decodes['____AMPERSAND____'] = '&';
        $decodes['____BACKSLASH____'] = '\\';
        return str_replace(array_keys($decodes), array_values($decodes), $param);
    }

    function getChartCategoryBase($category)
    {
        return ARTIOFUSIONCHARTS_BASE_ABS . DS . $category;
    }

    function getChartMainCategoryParams($category)
    {
        return ARTIOFusionChartsHelper::getChartCategoryBase($category) . DS . 'params.php';
    }

    function getChartBaseAbs($type, $item)
    {
        return ARTIOFUSIONCHARTS_BASE_ABS . DS . $type . DS . $item . DS;
    }

    function getChartBaseUrl($type, $item)
    {
        return ARTIOFUSIONCHARTS_BASE_URL . '/' . $type . '/' . $item . '/';
    }

    function getChartIcon($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseUrl($type, $item) . 'icon.png';
    }

    function getChartPrewiev($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseUrl($type, $item) . 'preview.jpg';
    }

    function getChartFile($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseUrl($type, $item) . 'chart.swf';
    }

    function getChartParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'params.xml';
    }

    function getChartCategoriesParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'params-categories.xml';
    }

    function getChartCategoryParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'params-category.xml';
    }

    function getChartDatasetParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'params-dataset.xml';
    }

    function getChartSetParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'params-set.xml';
    }

    function getChartTrendLinesParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'trendlines.xml';
    }

    function getChartMainParams($type, $item)
    {
        return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'params.php';
    }
    
    function getChartLineSetParams($type, $item)
    {
    	return ARTIOFusionChartsHelper::getChartBaseAbs($type, $item) . 'lineset.xml';
    }
}

?>