<?php

/**
 * Reservation model. Support for database operations.
 * 
 * @version		$Id$
 * @package		ARTIO Booking
 * @subpackage  models
 * @copyright	Copyright (C) 2010 ARTIO s.r.o.. All rights reserved.
 * @author 		ARTIO s.r.o., http://www.artio.net
 * @license     GNU/GPL http://www.gnu.org/copyleft/gpl.html
 * @link        http://www.artio.net Official website
 */

defined('_JEXEC') or die('Restricted access');

//import needed JoomLIB helpers
AImporter::helper('booking', 'model');

class BookingModelReservation extends AModel
{
    
    /**
     * Main table
     * 
     * @var TableReservation
     */
    var $_table;

    function __construct()
    {
        parent::__construct();
        $this->_table = $this->getTable('reservation');
    }
    
    /**
     * (non-PHPdoc)
     * @see AModel::getObject()
     */
    function getObject()
    {
    	$object = parent::getObject();
    	/* @var $object TableReservation */
    	$object->creator = '';
    	$object->modifier = '';
    	
    	if ($this->_id) {
    		$query = 'SELECT c.name AS creator, m.name AS modifier ';
    		$query .= 'FROM #__booking_reservation AS r ';
    		$query .= 'LEFT JOIN #__users AS c ON r.created_by = c.id ';
	    	$query .= 'LEFT JOIN #__users AS m ON r.modified_by = m.id ';
	    	$query .= 'WHERE r.id = ' . (int) $this->_id;
	    	$this->_db->setQuery($query);
	    	$data = $this->_db->loadObject();
	    	
	    	$object->creator = $data->creator;
	    	$object->modifier = $data->modifier;
    	}
    	
    	return $object;
    }

    /**
     * Save reservation.
     * 
     * @param array $data request data
     * @return customer id if success, false in unsuccess
     */
    function store($data)
    {
    	$data['book_time'] = date("Y-m-d H:i:s");
        if (! $this->_table->bind($data) || ! $this->_table->check() || ! $this->_table->store()) {
            return false;
        }
        
        return $this->_table->id;
    }

    /**
     * Set reservations as received.
     * 
     * @param array $cids
     * @return boolean
     */
    function receive($cids)
    {
        return $this->state('paid', $cids, RESERVATION_RECEIVE, RESERVATION_RECEIVE_DEPOSIT, RESERVATION_PENDING);
    }

    /**
     * Set reservations as received deposit.
     * 
     * @param array $cids
     * @return boolean
     */
    function receiveDeposit($cids)
    {
        return $this->state('paid', $cids, RESERVATION_RECEIVE_DEPOSIT, RESERVATION_PENDING, RESERVATION_RECEIVE);
    }

    /**
     * Set reservations as unreceived.
     * 
     * @param array $cids
     * @return boolean
     */
    function unreceive($cids)
    {
        return $this->state('paid', $cids, RESERVATION_PENDING, RESERVATION_RECEIVE, RESERVATION_RECEIVE_DEPOSIT);
    }

    /**
     * Set reservations as storned.
     * 
     * @param array $cids
     * @return boolean
     */
    function storno($cids)
    {
        return $this->state('state', $cids, RESERVATION_STORNED, RESERVATION_ACTIVE, RESERVATION_TRASHED);
    }

    /**
     * Safe storno reservations. Control customer ID and current state.
     * Storno only active reservations.
     * Storno only reservations which haveno item expired.
     * 
     * @param int $customerId
     * @param array $cids
     * @return boolean
     */
    function stornoSafe($customerId, $cids)
    {
        if (count($cids)) {
        	
        	$return = true;
        	
        	$modelReservationItems = new BookingModelReservationItems();
        	$modelReservationItems->init(array());	
        	
        	$now = AModel::getNow();
        	
        	foreach ($cids as $cid) {
        		
        		$isExpired = false;
        		
        		$modelReservationItems->_lists['reservation_item-reservation_id']=$cid;
        		unset($modelReservationItems->_data);
        		$reservedItems = $modelReservationItems->getData();
        		
        		foreach ($reservedItems as $reservedItem)
        			if (strtotime($reservedItem->from) <= $now)
        				$isExpired=true;

        		if (!$isExpired) { //no item in reservation is expired
	        				
		            $query = 'UPDATE `' . $this->_table->getTableName() . '` SET `state` = ' . RESERVATION_STORNED;
		            $query .= ' WHERE `customer` = ' . (int) $customerId . ' AND `state` = ' . RESERVATION_ACTIVE;
		            $query .= ' AND `id` = ' . (int)$cid;
		            $this->_db->setQuery($query);
		            $return = $this->_db->query();
        		}
        	}
            return $return;
        }
        return false;
    }

    /**
     * Set reservations as trashed.
     * 
     * @param array $cids
     * @return boolean
     */
    function trash($cids)
    {
        return $this->state('state', $cids, RESERVATION_TRASHED, RESERVATION_ACTIVE, RESERVATION_STORNED, RESERVATION_CONFLICTED);
    }
    
    /**
     * Set reservations as conflicted.
     *
     * @param array $cids
     * @return boolean
     */
    function conflict($cids)
    {
    	return $this->state('state', $cids, RESERVATION_CONFLICTED, RESERVATION_ACTIVE, RESERVATION_STORNED, RESERVATION_TRASHED, RESERVATION_PRERESERVED);
    }
    
    /**
     * Set reservations as pre-reserved.
     *
     * @param array $cids
     * @return boolean
     */
    function prereserved($cids)
    {
    	return $this->state('state', $cids, RESERVATION_PRERESERVED, RESERVATION_ACTIVE, RESERVATION_STORNED, RESERVATION_TRASHED, RESERVATION_CONFLICTED);
    }

    /**
     * Set reservations as actived.
     * 
     * @param $cids
     */
    function active($cids)
    {
    	$error = false;
    	foreach($cids as $id)
    	{
    	
    		//get reservation items for reservation
	    	$lists['reservation_item-reservation_id'] = $id;	        
	    	$modelReservations = new BookingModelReservationItems();
	    	$modelReservations->init($lists);
	    	$items = $modelReservations->getData();
	    	
	    	$modelSubject = new BookingModelSubject();
	    	
	    	$conflict = false;
	    	//process all items for current reservation
	    	foreach($items as $item)
	    	{
				//get all reservations for subject in same time interval
	    		$reservations = $modelReservations->getSimpleData($item->subject,$item->from,$item->to,true);
	    		
	    		//get total reserved capacity for subject in time interval
	    		$sum = 0;
	    		foreach($reservations as $reservation)
	    		{
	    			$sum += $reservation->capacity;
	    		}
	    		
	    		//get subject's total capacity and compare with sum
	    		if($modelSubject->setId($item->subject)->getObject()->total_capacity < $sum)
	    		{
	    			$conflict = true;
	    			$error = true;
	    			break;
	    		}
	    	}
	    	if($conflict){
	    		JLog::getInstance()->addEntry(array('comment'=>"Conflict with capacity for reservation ".$id));
	    		$this->conflict(array($id));
	    	}
    	}
    	
    	if($error){
    		return false;
    	}

        return $this->state('state', $cids, RESERVATION_ACTIVE, RESERVATION_STORNED, RESERVATION_TRASHED, RESERVATION_CONFLICTED, RESERVATION_PRERESERVED);
    }

    /**
     * Set reservations as actived (untrashed).
     * 
     * @param array $cids
     * @return boolean
     */
    function restore($cids)
    {
        return $this->state('state', $cids, RESERVATION_ACTIVE, RESERVATION_TRASHED);
    }

    /**
     * Delete trashed reservations.
     * 
     * @return boolean
     */
    function emptyTrash()
    {
    	$this->_db->setQuery('DELETE r, i, p, s 
    			FROM #__booking_reservation AS r 
    			LEFT JOIN #__booking_reservation_items AS i ON r.id = i.reservation_id
    			LEFT JOIN #__booking_reservation_period AS p ON p.reservation_item_id = i.id 
    			LEFT JOIN #__booking_reservation_supplement AS s ON r.id = s.reservation 
    			WHERE r.state = ' . RESERVATION_TRASHED);
    	$this->_db->query();
    	return true;
    }

    function setPaymentType($id, $alias, $title)
    {
        $this->_db->setQuery('UPDATE `' . $this->_table->getTableName() . '` SET `payment_method_id` = ' . $this->_db->Quote($alias) . ',`payment_method_name` = ' . $this->_db->Quote($title) . ' WHERE `id` = ' . (int) $id);
        return $this->_db->query();
    }
    
    /**
     * Get full price for paing.
     *
     * @return int
     */
    function getFullPrice()
    {
    	return $this->_table->fullPrice;
    }
    
    /**
     * Get price when deposit is paid.
     *
     * @return int
     */
    function getFullPriceWithoutDeposit()
    {
    	return $this->_table->fullPrice - $this->_table->fullDeposit;
    }
    
    function followup() 
    {
    	$config = AFactory::getConfig();
    	/* @var $config BookingConfig */
    	
    	if (!$config->followupEnabled)
    		die(JText::_('Follow Up Disabled'));
    	
    	$match = array();
    	if (preg_match('/^(\d+)t(\d+)d(\d+)h(\d+)m$/', $config->followupSchedule, $match)) {
    		$type = $match[1];
    		$days = $match[2];
    		$hours = $match[3];
    		$minutes = $match[4];
    	} else
    		die(JText::_('Wrong Follow Up Schedule'));
    	
    	$email = JModel::getInstance('Email', 'BookingModel')->getItem($config->followupEmail);
    	if (!$email->id)
    		die('E-mail Template not Set');
    	
    	$date = $this->_db->quote(gmdate('Y-m-d H:i:s', JFactory::getDate()->toUnix() - $days * 86400 - $hours * 3600 - $minutes * 60));

    	$query = 'SELECT r.id, r.email FROM #__booking_reservation AS r ';
    	$query .= 'JOIN #__booking_reservation_items AS i ';
    	$query .= 'WHERE `to` <= ' . $date . ' AND state = ' . RESERVATION_ACTIVE . ' AND paid = ' . RESERVATION_RECEIVE . ' AND follow_up_sent = 0 ';
    	$query .= 'GROUP BY r.id';   		
    	
    	$this->_db->setQuery($query);
    	$items = $this->_db->loadObjectList();
    	
    	$ids = array();
    	
    	foreach ($items as $item) {
    		JUtility::sendMail(JFactory::getApplication()->getCfg('mailfrom'), JFactory::getApplication()->getCfg('fromname'), $item->email, $email->subject, $email->body, $email->mode);
    		$ids[] = $item->id;
    	}
    	
    	if (count($ids)) {
    		$this->_db->setQuery('UPDATE #__booking_reservation SET follow_up_sent = 1 WHERE id IN (' . implode(', ', $ids) . ')');
    		$this->_db->query();
    		die(count($ids).' e-mails sent');
    	}
    		
    	die();
    }
}

?>