<?php
/**
 * SEF component for Joomla!
 * 
 * @package   JoomSEF
 * @version   4.7.10
 * @author    ARTIO s.r.o., http://www.artio.net
 * @copyright Copyright (C) 2021 ARTIO s.r.o. 
 * @license   GNU/GPLv3 http://www.artio.net/license/gnu-general-public-license
 */

// Check to ensure this file is included in Joomla!
defined('_JEXEC') or die();

class SefSitemapXmlGenerator implements ISefSitemapXmlGenerator
{
    const MAX_SITEMAP_URLS = 50000;

    /** @var string */
    protected $filename;
    /** @var ISefUrlDomainProvider */
    protected $domainProvider;
    /** @var string */
    protected $sitemapDomain;

    /** @var SefSitemapXmlWriter */
    protected $sitemapXml;
    /** @var SefSitemapIndexXmlWriter */
    protected $sitemapIndexXml;

    protected $xmlCount = 0;

    /**
     * @param string $filename
     * @param ISefUrlDomainProvider $urlDomainProvider
     * @param string $sitemapDomain
     */
    public function __construct($filename, $urlDomainProvider, $sitemapDomain)
    {
        $this->filename = $filename;
        $this->domainProvider = $urlDomainProvider;
        $this->sitemapDomain = $sitemapDomain;
    }

    public function open()
    {
        if ($this->sitemapXml)
            throw new RuntimeException(JText::_('COM_SEF_ERROR_SAVE_XML'));

        $this->openSitemapXml();
    }

    protected function prepareNextSitemapXml()
    {
        $this->sitemapXml->close();

        if (!$this->sitemapIndexXml) {
            // First sitemap XML, rename it and create sitemap index XML file
            $oldFile = $this->sitemapXml->getFilePath();
            $newFile = $this->getFullFilename($this->xmlCount);
            if (!@rename($oldFile, $newFile))
                throw new RuntimeException(JText::_('COM_SEF_ERROR_SAVE_XML'));

            $this->sitemapIndexXml = new SefSitemapIndexXmlWriter($oldFile);
        }

        $this->sitemapIndexXml->addSitemap($this->getSitemapUrl($this->xmlCount));

        $this->openSitemapXml();
    }

    protected function openSitemapXml()
    {
        $fullName = $this->getFullFilename($this->xmlCount > 0 ? ($this->xmlCount + 1) : '');
        $this->sitemapXml = new SefSitemapXmlWriter($fullName);
        $this->xmlCount++;
    }

    /**
     * @param string $index
     * @return string
     */
    protected function getFullFilename($index)
    {
        return JPATH_ROOT.'/'.$this->filename.$index.'.xml';
    }

    /**
     * @param string $index
     * @return string
     */
    protected function getSitemapUrl($index)
    {
        return $this->sitemapDomain.$this->filename.$index.'.xml';
    }

    /**
     * @param SefSitemapUrl $url
     */
    public function addUrl($url)
    {
        if (!$this->sitemapXml)
            throw new RuntimeException(JText::_('COM_SEF_ERROR_SAVE_XML'));

        if ($this->sitemapXml->getUrlsCount() >= self::MAX_SITEMAP_URLS) {
            $this->prepareNextSitemapXml();
        }

        $this->sitemapXml->addUrl($this->domainProvider->getUrlDomain($url), $url);
    }

    public function close()
    {
        if (!$this->sitemapXml)
            throw new RuntimeException(JText::_('COM_SEF_ERROR_SAVE_XML'));

        $this->sitemapXml->close();

        if ($this->sitemapIndexXml) {
            $this->sitemapIndexXml->addSitemap($this->getSitemapUrl($this->xmlCount));
            $this->sitemapIndexXml->close();
        }
    }

    /**
     * @return string[]
     */
    public function getSitemapsUrls()
    {
        return array(
            $this->getSitemapUrl('')
        );
    }
}
