<?php

/**
 * @version		$Id$
 * @package		Joomla.Administrator
 * @subpackage	JoomDOC
 * @author      ARTIO s.r.o., info@artio.net, http:://www.artio.net
 * @copyright	Copyright (C) 2011 Artio s.r.o.. All rights reserved.
 * @license		GNU General Public License version 2 or later; see LICENSE.txt
 */

jimport('joomla.filesystem.archive');

class JoomDOCFileSystem {

    /**
     * Get informations about folders and files in folder.
     *
     * @param string $absolutePath absolute folder path
     * @param string $filter filter folder/file name
     * @return JoomDOCFolder if path is not exists folder function return false
     */
    public static function getFolderContent ($absolutePath, $filter) {
        if (JFolder::exists($absolutePath)) {
            $root = new JoomDOCFolder($absolutePath);
            foreach (JFolder::folders($absolutePath, $filter . '.', false, true) as $folder)
                $root->addFolder(new JoomDOCFolder($folder));
            foreach (JFolder::files($absolutePath, $filter . '.', false, true) as $file)
                $root->addFile(new JoomDOCFile($file));
            return $root;
        } elseif (JFile::exists($absolutePath))
            return new JoomDOCFile($absolutePath);
        return false;
    }

    /**
     * Convert absolute filepath in file system to relative path from Joomla root.
     *
     * @param string $absolutePath absolute path to convert
     * @return string relative path
     */
    public static function getRelativePath ($absolutePath) {
        $config = JoomDOCConfig::getInstance();
        if ($config->docroot == $absolutePath)
            return '';
        return str_replace($config->docroot . DS, '', $absolutePath);
    }

    /**
     * Get file URL.
     *
     * @param string $relativePath file relative path from document root
     * @return string
     */
    public function getURL ($relativePath) {
        $config = JoomDOCConfig::getInstance();
        return JURI::root() . $config->docrootrel . JPath::clean('/' . $relativePath, '/');
    }

    /**
     * Get last item from path. File or last folder name.
     *
     * @param string $absolutePath absolute path
     * @return string name of last folder or file, if not found function return false
     */
    public static function getLastPathItem ($absolutePath) {
        if (is_array(($parts = explode(DS, $absolutePath))))
            return end($parts);
        return false;
    }

    /**
     * Get absolute path bread crumbs from document root.
     *
     * @param string $absolutePath absolute path
     * @return array key is full absolute path of crumb, value is last folder name in crumb, if not found function return false
     */
    public function getPathBreadCrumbs ($absolutePath) {
        $config = JoomDOCConfig::getInstance();
        if (is_array(($parts = explode(DS, str_replace($config->docroot, '', JPath::clean($absolutePath)))))) {
            $breadCrumbs[($root = $config->docroot)] = JText::_('JOOMDOC_ROOT');
            foreach ($parts as $part)
                if ($part)
                    $breadCrumbs[($root .= DS . $part)] = $part;
            return $breadCrumbs;
        }
        return false;
    }

    /**
     * Create new folder in parent folder.
     *
     * @param string $parentFolder parent folder absolute path
     * @param string $newFolder new folder name without path, if $parentFolder is null use $newFolder as full path
     * @return boolean if true success. If failed function return error message as string.
     */
    public function newFolder ($parentFolder, $newFolder) {
        if ($parentFolder) {
            $absolutePath = JPath::clean($parentFolder . DS . $newFolder);
        } else {
            $absolutePath = $newFolder;
        }
        if ($parentFolder && !JFolder::exists($parentFolder))
            JError::raiseWarning(21, JText::sprintf('JOOMDOC_PARENT_FOLDER_NO_EXISTS', $parentFolder));
        elseif (!($newFolder = JString::trim($newFolder)))
            JError::raiseWarning(21, JText::sprintf('JOOMDOC_NEW_FOLDER_CANNOT_HAVE_EMPTY_NAME', $absolutePath));
        elseif ($parentFolder && JString::strpos($newFolder, DS) !== false)
            JError::raiseWarning(21, JText::sprintf('JOOMDOC_FOLDER_NAME_CANNOT_CONTAIN_SLASH', $absolutePath));
        elseif (JFolder::exists($absolutePath))
            JError::raiseWarning(21, JText::sprintf('JOOMDOC_FOLDER_ALREADY_EXISTS', $absolutePath));
        else {
            if (JFolder::create($absolutePath)) {
                $table =& JTable::getInstance('File', 'JoomDOCTable');
                /* @var $table JoomDOCTableFile */
                $table->path = JoomDOCFileSystem::getRelativePath($absolutePath);
                $table->store();
                return 'JOOMDOC_FOLDER_CREATED';
            }
            JError::raiseWarning(21, JText::sprintf('JOOMDOC_UNABLE_CREATE_FOLDER', $absolutePath));
        }
        return false;
    }

    /**
     * Test if folder is subfolder of its parent folder.
     *
     * @param string $subfolder absolute path to folder which we need test
     * @param string $folder absolute path to tested folder parent folder
     * @return mixed true/false/null - is subfolder/is not subfolder/unable test
     */
    public function isSubFolder ($subfolder, $folder) {
        $subfolder = JPath::clean($subfolder);
        $folder = JPath::clean($folder);
        if (!JoomDOCFileSystem::exists($subfolder) || !JoomDOCFileSystem::exists($folder)) {
            return null;
        }
        return JString::strpos($subfolder, $folder) === 0;
    }

    /**
     * File or folder exists.
     *
     * @param string $path absolute path
     * @return boolean true/false - exists/no exists
     */
    public function exists ($path) {
        return JFolder::exists($path) || JFile::exists($path);
    }

    /**
     * Delete folder or file.
     *
     * @param string $path absolute path
     * @return boolean true/false - success/unsuccess
     */
    public function deleteItem ($path) {
        return (((JFolder::exists($path) && JFolder::delete($path)) || (JFile::exists($path) && JFile::delete($path))));
    }

    /**
     * Delete folders/files.
     *
     * @param mixed $path paths to delete
     * @return void
     */
    public function delete ($paths) {
        if (!is_array($paths)) {
            $paths = array($paths);
        }
        $count = 0;
        $table =& JTable::getInstance(JOOMDOC_FILE, JOOMDOC_TABLE_PREFIX);
        /* @var $table JoomDOCTableFile */
        $paths = array_map('JoomDOCFileSystem::getRelativePath', $paths);
        $tree =& $table->delete($paths);
        foreach ($tree as $item) {
                        if (JoomDOCFileSystem::deleteItem(JoomDOCFileSystem::getFullPath($item))) {
                $count++;
            }
        }
        return $count;
    }

    /**
     * Get file size in human readable format.
     *
     * @param string $absolutePath file absolute path
     * @return string file size with unit
     */
    function getFileSize ($absolutePath) {
        if (JFile::exists($absolutePath)) {
            // function sometimes provide warning - for this reason is used @
            $filesize = @filesize($absolutePath);
            if ($filesize) {
                if ($filesize >= 1000000000)
                    return JText::sprintf('JOOMDOC_GB_SIZE', round($filesize / 1000000000, 1));
                elseif ($filesize >= 1000000)
                    return JText::sprintf('JOOMDOC_MB_SIZE', round($filesize / 1000000, 1));
                elseif ($filesize >= 1000)
                    return JText::sprintf('JOOMDOC_KB_SIZE', round($filesize / 1000, 1));
                return JText::sprintf('JOOMDOC_B_SIZE', round($filesize), 1);
            }
        }
        return '-';
    }

    /**
     * Upload file or zipe archive.
     *
     * @param string $folder destination folder absolute path
     * @param string $field request data name
     * @param boolean $isZip if true file will unpacked in destination folder
     * @return void
     */
    function upload ($folder, $field, $isZip) {
        $mainframe =& JFactory::getApplication();
        /* @var $mainframe JApplication */
        $count = 0;
        $data =& JRequest::getVar($field, null, 'files', 'array');
        if ($data) {
            if ($data['error'] != 0)
                JError::raiseWarning(21, JText::sprintf('JOOMDOC_UNABLE_UPLOAD_FILE', ''));
            elseif (!JFolder::exists($folder))
                JError::raiseWarning(21, JText::sprintf('JOOMDOC_PARENT_FOLDER_NO_EXISTS'), $folder);
            else {
                if ($isZip && ($tmpFolder = JoomDOCFileSystem::createTemporaryFolder('joomdoc_unpack'))) {
                    $zip =& JArchive::getAdapter('zip');
                    /* @var $zip JArchiveZip */
                    if ($zip->extract($data['tmp_name'], $tmpFolder) !== true)
                        JError::raiseWarning(21, JText::sprintf('JOOMDOC_UNABLE_EXTRACT_FILE', $data['name']));
                    else {
                        foreach (JFolder::folders($tmpFolder, '.', true, true) as $zipFolder)
                            if (!JFolder::exists(($path = JPath::clean($folder . DS . str_replace($tmpFolder, '', $zipFolder)))))
                                JoomDOCFileSystem::newFolder(null, $path);
                        foreach (JFolder::files($tmpFolder, '.', true, true) as $zipFile)
                            if (JoomDOCFileSystem::uploadFile($folder, $zipFile, str_replace($tmpFolder . DS, '', $zipFile), true))
                                $count++;
                    }
                    JFolder::delete($tmpFolder);
                } elseif (JoomDOCFileSystem::uploadFile($folder, $data['tmp_name'], $data['name']))
                    $count++;
            }
        }
        return $count;
    }

    /**
     * Upload file in folder from source destination.
     *
     * @param string $folder absolute path to destination folder
     * @param string $filenameSource source file path
     * @param string $filenameDest destnation file name
     * @param boolean $copy use copy function instead upload, default false
     * @return boolean
     */
    private function uploadFile ($folder, $filenameSource, $filenameDest, $copy = false) {
        $config =& JoomDOCConfig::getInstance();
        $absolutePath = $folder . DS . $filenameDest;
        $table =& JTable::getInstance(JOOMDOC_FILE, JOOMDOC_TABLE_PREFIX);
        /* @var $table JoomDOCTableFile */
        $table->path = JoomDOCFileSystem::getRelativePath($absolutePath);
                $upload = $copy ? JFile::copy($filenameSource, $absolutePath) : JFile::upload($filenameSource, $absolutePath);
        if ($upload) {
                        
             $table->store();
             
        }
        return $upload;
    }
        /**
     * Get temporary folder. Use setting from Joomla! global configuration.
     *
     * @return string folder absolute path, if folder no exists or is unwritable function return false
     */
    public function getJoomlaTemporaryFolder () {
        $mainframe =& JFactory::getApplication();
        /* @var $mainframe JApplication */
        if (!JFolder::exists(($tmpFolder = $mainframe->getCfg('tmp_path'))))
            JError::raiseNotice(21, JText::sprintf('JOOMDOC_JOOMLA_TEMPORARY_FOLDER_NO_EXISTS', $tmpFolder));
        elseif (!is_writable($tmpFolder))
            JError::raiseNotice(21, JText::sprintf('JOOMDOC_JOOMLA_TEMPORARY_FOLDER_IS_UNWRITABLE', $tmpFolder));
        else
            return $tmpFolder;
        if (JFolder::exists(($tmpFolder = JPATH_ROOT . DS . 'tmp')) && is_writable($tmpFolder))
            return $tmpFolder;
        return false;
    }

    /**
     * Create temporary folder in Joomla! template directory.
     *
     * @param string $mask name of new folder if folder already exist function given number on name begin
     * @return string template folder absolute path, if unable folder create function return false
     */
    public function createTemporaryFolder ($mask) {
        if (($tmpFolder = JoomDOCFileSystem::getJoomlaTemporaryFolder())) {
            $r = '';
            do {
                $absolutePath = JPath::clean($tmpFolder . DS . ($r++) . $mask);
            } while (JFolder::exists($absolutePath));
            if (!JFolder::create($absolutePath))
                JError::raiseWarning(21, JText::sprintf('JOOMDOC_UNABLE_CREATE_TEMPLATE_FOLDER', $absolutePath));
            else
                return $absolutePath;
        }
        return false;
    }

    /**
     * Get path of parent folder from path.
     *
     * @param string $path
     * @return string
     */
    public function getParentPath ($path) {
        if (($pos = JString::strrpos($path, DS)) === false)
            return null;
        return JString::substr($path, 0, $pos);
    }

    /**
     * Item is folder or no.
     *
     * @param mixed $item file/folder instance
     * @return boolean
     */
    public function isFolder (&$item) {
        return ($item instanceof JoomDOCFolder);
    }

    /**
     * Item is file or no.
     *
     * @param mixed $item file/folder instance
     * @return boolean
     */
    public function isFile (&$item) {
        return ($item instanceof JoomDOCFile);
    }

    /**
     * Get full absolute path.
     *
     * @param string $path relative path
     * @return string
     */
    public function getFullPath ($path) {
        $config = JoomDOCConfig::getInstance();
        return JPath::clean($config->docroot . DS . $path);
    }

    /**
     * Download file by request param path (relative path from doc root).
     *
     * @param booelan $saveHits if true increment file download counter
     * @return void
     */
    public function download ($saveHits = true) {
        $mainframe =& JFactory::getApplication();
        /* @var $mainframe JApplication */
        $modelDocument =& JModel::getInstance(JOOMDOC_DOCUMENT, JOOMDOC_MODEL_PREFIX);
        /* @var $modelDocument JoomDOCModelDocument */
        $modelFile =& JModel::getInstance(JOOMDOC_FILE, JOOMDOC_MODEL_PREFIX);
        /* @var $modelFile JoomDOCModelFile */

        // get file path from request
        $path = JoomDOCString::urldecode(JString::trim(JRequest::getString('path')));
        
        // path can be document full alias
        $candidate = $modelDocument->searchRelativePathByFullAlias($path);

        if ($candidate) {
            $path = $candidate;
        }

        // get document by file relative path
        $document =& $modelDocument->getItemByPath($path);
                
         $fullPath = JoomDOCFileSystem::getFullPath($path);
         
        // open file to reading
        $content = JFile::read($fullPath);

        if (($document && $document->published == JOOMDOC_STATE_UNPUBLISHED) || !JFile::exists($fullPath) || $content === false) {
            // file document unpublish or file doesn't exists or cannot read
            JError::raiseWarning(404, JText::_('JOOMDOC_FILE_NO_AVAILABLE'));
            $mainframe->redirect(JoomDOCRoute::viewDocuments());
            return;
        }

        if ($saveHits) {
            // save file downloading
                        
            $modelFile->saveHits($path);
            
        }

        // clean output
        ob_clean();

        // prepare packet head
        if (function_exists('mime_content_type')) {
            header('Content-Type: ' . mime_content_type($fullPath) . '; charset=UTF-8');
        }
        header('Content-Transfer-Encoding: 8bit');
        header('Content-Disposition: attachment; filename="' . JFile::getName($fullPath) . '";');
        $fileSize = @filesize($fullPath);
        if ($fileSize) {
            header('Content-Length: ' . $fileSize);
        }
        // flush file
        die($content);
    }
        /**
     * Rename file/folder and versions folder. With failed roll back changes.
     * Rename path in database (file/document).
     *
     * @param string $oldPath relative path of file/folder to rename
     * @param string $newName new file/folder mame (without path)
     * @return boolean true/false - success/unsuccess
     */
    function rename ($oldPath, $newName) {
        // parent folder of affected file/folder
        $parentPath = JoomDOCFileSystem::getParentPath($oldPath);
        if ($parentPath) {
            $newPath = $parentPath . DS . $newName;
        } else {
            $newPath = $newName;
        }
        // old absolute path of affected file/folder
        $oldPathFull = JoomDOCFileSystem::getFullPath($oldPath);
        // new absolute path of affected file/folder
        $newPathFull = JoomDOCFileSystem::getFullPath($newPath);
                // try to rename file or folder
        if ((JFile::exists($oldPathFull) && !JFile::move($oldPathFull, $newPathFull)) || (JFolder::exists($oldPathFull) && !JFolder::move($oldPathFull, $newPathFull))) {
            // unable rename
            return false;
        }
                // rename in database
        $table =& JTable::getInstance(JOOMDOC_FILE, JOOMDOC_TABLE_PREFIX);
        /* @var $table JoomDOCTableFile */
        $table->rename($oldPath, $newPath);
                return true;
    }
}
?>