#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdbool.h>
#include "conv.h"

int main( int argc, char* argv[] )
{
  if( (argc == 2) && (strcmp(argv[1], "-h") == 0) )
  {
    PrintHelp();
    return EXIT_SUCCESS;
  }
  else if( argc == 3 )
  {
    int result = FixFile( argv[1], argv[2] );
    if( result != E_OK )
    {
      PrintError( result );
      return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
  }

  PrintError( E_CMDLINE );
  return EXIT_FAILURE;
}

void PrintHelp()
{
  fprintf( stdout,
           "Utility: conv\n"
           "\n"
           "Conv will transform URLs of JoomSEF 1.2.5 to format\n"
           "compatible with JoomSEF 1.3.1. Supports SQL dumps and\n"
           "JoomSEF export files.\n"
           "\n"
           "Usage: conv -h\n"
           "       conv infile outfile\n"
           "\n"
           "Parameter:\n"
           "  -h       Display this help text\n"
           "  infile   Path to file to be converted\n"
           "  outfile  Path where to store converted file\n" );
}

void PrintError( int code )
{
  switch( code )
  {
  case E_CMDLINE:
    fprintf( stderr,
             "The program was called with wrong arguments.\n"
             "To show help run conv -h.\n" );
    break;

  case E_FILEIN:
    fprintf( stderr,
             "Cannot open input file for reading.\n" );
    break;

  case E_FILEOUT:
    fprintf( stderr,
             "Cannot open output file for writing.\n" );
    break;

  case E_OUTOFMEMORY:
    fprintf( stderr,
             "Cannot allocate more memory. Program will be terminated.\n" );
    break;
  }
}

int FixFile( char* szIn, char* szOut )
{
  FILE* fIn = fopen( szIn, "rt" );
  if( !fIn )
    return E_FILEIN;

  FILE* fOut = fopen( szOut, "wt" );
  if( !fOut )
  {
    fclose( fIn );
    return E_FILEOUT;
  }

  char* szLine = NULL;
  unsigned int iLen = 0;

  while( !feof(fIn) )
  {
    ReadLine(&szLine, &iLen, fIn);
      
    char* firstPart = szLine;
    char* pos = strstr(szLine, "'index.php?");
    if( pos == NULL )
    {
      fprintf( fOut, "%s\n", szLine );
      free(szLine);
      szLine = NULL;
      iLen = 0;
      continue;
    }
    pos += 11;  // Nechme si pouze cast za index.php?
    char* url = (char*)malloc(sizeof(char)*iLen);
    if( !url )
      return E_OUTOFMEMORY;
    strcpy(url, pos);
    *pos = '\0';

    pos = strstr(url, "'");
    if( pos == NULL )
    {
      free(url);
      free(szLine);
      szLine = NULL;
      iLen = 0;
      continue;
    }
    char* lastPart = (char*)malloc(sizeof(char)*iLen);
    if( !lastPart )
      return E_OUTOFMEMORY;
    strcpy(lastPart, pos);
    *pos = '\0';

    int result = Sort( url );
    if( result != E_OK )
      return result;

    fprintf( fOut, "%s%s%s\n", firstPart, url, lastPart );

    free(lastPart);
    free(url);
    free(szLine);
    szLine = NULL;
    iLen = 0;
  }

  fclose( fOut );
  fclose( fIn );

  return E_OK;
}

// Jednoduchm bubble sortem
int Sort( char* szUrl )
{
  tVar** vars = NULL;
  int nVars = 0;

  int result = GetVars( szUrl, &vars, &nVars );
  if( result != E_OK )
    return result;

  bool bChange;
  do
  {
    bChange = false;
    for( int i = 0; i < nVars-1; i++ )
    {
      if( strcmp(vars[i]->varName, vars[i+1]->varName) > 0 )
      {
        bChange = true;
        tVar* tmp = vars[i];
        vars[i] = vars[i+1];
        vars[i+1] = tmp;
      }
    }
  } while (bChange);

  ComposeUrl( szUrl, vars, nVars );

  FreeVars( vars, nVars );

  return E_OK;
}

int GetVars( char* szUrl, tVar*** vars, int* nVars )
{
  int count = 0;
  int capacity = 4;
  tVar** pvars = (tVar**)malloc(sizeof(tVar*)*capacity);
  if( !pvars )
    return E_OUTOFMEMORY;
  
  char* str = szUrl;
  while( str )
  {
    tVar* newVar = (tVar*)malloc(sizeof(tVar));
    if( !newVar )
      return E_OUTOFMEMORY;
    char* pos = strstr(str, "=");
    *pos = '\0';
    newVar->varName = (char*)malloc(sizeof(char)*(strlen(str)+1));
    if( !(newVar->varName) )
      return E_OUTOFMEMORY;
    strcpy(newVar->varName, str);
    str=pos+1;
    pos = strstr(str, "&");
    if( pos != NULL )
      *pos = '\0';
    newVar->varValue = (char*)malloc(sizeof(char)*(strlen(str)+1));
    if( !(newVar->varValue) )
      return E_OUTOFMEMORY;
    strcpy(newVar->varValue, str);
    str=pos+1;
    if( pos == NULL )
      str = NULL;

    if( count == capacity )
    {
      capacity *= 2;
      pvars = (tVar**)realloc(pvars, sizeof(tVar*)*capacity);
    }
    pvars[count] = newVar;
    count++;
  }

  *vars = pvars;
  *nVars = count;

  return E_OK;
}

void ComposeUrl( char* szUrl, tVar** vars, int nVars )
{
  *szUrl = '\0';

  for( int i = 0; i < nVars; i++ )
  {
    strcat(szUrl, vars[i]->varName);
    strcat(szUrl, "=");
    strcat(szUrl, vars[i]->varValue);
    if( i != nVars-1 )
      strcat(szUrl, "&");
  }
}

void FreeVars( tVar** vars, int nVars )
{
  for( int i = 0; i < nVars; i++ )
  {
    free(vars[i]->varName);
    free(vars[i]->varValue);
    free(vars[i]);
  }
  free(vars);
}

/**
 * Otestuje, zda posledni alokacni operace probehla v poradku. Pokud ne, 
 * vypise chybovou hlasku a ukonci program s chybovym kodem.
 */
inline void TestAlloc( void* ptr )
{
  if( ptr == NULL )
  {
    // vypise na standardni chybovy vystup chybovou hlasku.
    fprintf( stderr, "ERROR: Not enough memory!\n" );
    exit( EXIT_FAILURE );
  }
}

/**
 * Nacte radek ze standardniho vstupu. V nactenem retezci se nebude vyskytovat
 * znak konce radku '\n'. 
 */
unsigned int ReadLine( char** pLine, unsigned int* N, FILE* stream )
{
  // test, zda byly parametry predany odkazem v poradku
  assert( pLine != NULL );
  assert( N != NULL );
  assert( stream != NULL );
  
  const int B_INCREMENT = 16;
  int blockSize = *N;
  
  if ( *pLine == NULL )
  {
    // uzivatel nic nealokoval
    blockSize = B_INCREMENT;
    *pLine = (char*) malloc( blockSize * sizeof(char) );
    TestAlloc( *pLine );
  }
  
  int c;
  int i = 0;

  while( (c = getc(stream)) != EOF && c != '\n')
  {
    (*pLine)[i] = c;
    i += 1;
    if( (i % blockSize) == 0 )
    {
      // pokud je na konci bloku, je potreba jej natahnout
      blockSize += B_INCREMENT;
      *pLine = (char*) realloc( *pLine, blockSize );
      TestAlloc( *pLine );
    }
  }
  
  // Pozor! Pole predavane odkazem se musi indexovat takto. Bez zavorky
  // bychom indexovali pole ukazatelu, protoze operator * ma nizsi prioritu
  // nez [].
  (*pLine)[i] = '\0'; // na zaver je potreba retezec spravne ukoncit
  
  *N = blockSize;
  
  return i; // vraci delku retezce
}
